<?php

namespace App\Http\Controllers\Admin;

use App\Models\Frontend;
use App\Constants\Status;
use Illuminate\Http\Request;
use App\Rules\FileTypeValidate;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Storage;

class GeneralSettingController extends Controller
{
    private $user;
    public function __construct()
    {
        $this->user = auth()->guard('admin')->user();
        $this->check_permission("View - System Setting");
    }
    public function systemSetting()
    {
        $pageTitle = 'System Settings';
        $settings = json_decode(file_get_contents(resource_path('views/admin/setting/settings.json')));

        return view('admin.setting.system', compact('pageTitle', 'settings'));
    }

    public function general()
    {
        $pageTitle = 'General Setting';
        $timezones = timezone_identifiers_list();
        $currentTimezone = array_search(config('app.timezone'), $timezones);

        return view('admin.setting.general', compact('pageTitle', 'timezones', 'currentTimezone'));
    }

    public function generalUpdate(Request $request)
    {
        $request->validate([
            'site_name' => 'required|string|max:40',
            'cur_text' => 'required|string|max:40',
            'cur_sym' => 'required|string|max:40',
            'base_color' => 'nullable|regex:/^[a-f0-9]{6}$/i',
            'timezone' => 'required|integer',
            'currency_format' => 'required|in:1,2,3',
            'paginate_number' => 'required|integer',
            'show_number_after_decimal' => 'required|integer|gte:0|max:8',
            'first_exchange_bonus_percentage' => 'required|min:0',
            'exchange_auto_cancel_time' => 'required|gt:0',
            'register_bonus_amount' => 'required|numeric:gte:0',
            'exchange_lock_time' => 'required|numeric:gte:0',
        ]);
        

        $timezones = timezone_identifiers_list();
        $timezone = @$timezones[$request->timezone] ?? 'UTC';
        $general = gs();
        if($request->hasFile('exchange_alert_notification')){
                if($general->exchange_notification){
                    try {
                        unlink(public_path($general->exchange_notification));
                    } catch (\Throwable $th) {
                        Log::error('Exchange Notification audio file delete error');
                    }
                }
                $general->exchange_notification = $request->file('exchange_alert_notification')->store('assets/sound','public');
        }
        $previous_timezone = $general->timezone;
        $general->timezone = $request->timezone;

        $general->site_name = $request->site_name;
        $general->cur_text = $request->cur_text;
        $general->cur_sym = $request->cur_sym;
        $general->paginate_number = $request->paginate_number;
        $general->base_color = str_replace('#', '', $request->base_color);
        $general->currency_format = $request->currency_format;
        $general->show_number_after_decimal = $request->show_number_after_decimal;
        $general->first_exchange_bonus_percentage = $request->first_exchange_bonus_percentage;
        $general->exchange_auto_cancel_time = $request->exchange_auto_cancel_time;
        $general->register_bonus_amount = $request->register_bonus_amount;
        $general->exchange_lock_time = $request->exchange_lock_time;
        $general->save();

        $timezoneFile = config_path('timezone.php');
        $content = '<?php $timezone = "'.$timezone.'" ?>';
        file_put_contents($timezoneFile, $content);

        $notify[] = ['success', 'General setting updated successfully'];

        if($previous_timezone != $general->timezone){
            dispatch(function () use ($timezone) {
                setEnvValue('APP_TIMEZONE', $timezone);
                Artisan::call('config:clear');
            })->afterResponse();
        }

        return redirect()->back()->withNotify($notify);
    }

    public function systemConfiguration()
    {
        $pageTitle = 'System Configuration';

        return view('admin.setting.configuration', compact('pageTitle'));
    }

    public function systemConfigurationSubmit(Request $request)
    {
        $general = gs();
        $general->kv = $request->kv ? Status::ENABLE : Status::DISABLE;
        $general->ev = $request->ev ? Status::ENABLE : Status::DISABLE;
        $general->en = $request->en ? Status::ENABLE : Status::DISABLE;
        $general->sv = $request->sv ? Status::ENABLE : Status::DISABLE;
        $general->sn = $request->sn ? Status::ENABLE : Status::DISABLE;
        $general->pn = $request->pn ? Status::ENABLE : Status::DISABLE;
        $general->force_ssl = $request->force_ssl ? Status::ENABLE : Status::DISABLE;
        $general->secure_password = $request->secure_password ? Status::ENABLE : Status::DISABLE;
        $general->registration = $request->registration ? Status::ENABLE : Status::DISABLE;
        $general->agree = $request->agree ? Status::ENABLE : Status::DISABLE;
        $general->multi_language = $request->multi_language ? Status::ENABLE : Status::DISABLE;
        $general->show_notice_bar = $request->show_notice_bar ? Status::YES : Status::NO;
        $general->automatic_currency_rate_update = $request->automatic_currency_rate_update ? Status::YES : Status::NO;
        $general->admin_email_notification = $request->admin_email_notification ? Status::YES : Status::NO;
        $general->first_exchange_bonus = $request->first_exchange_bonus ? Status::YES : Status::NO;
        $general->exchange_auto_cancel = $request->exchange_auto_cancel ? Status::YES : Status::NO;
        $general->register_bonus = $request->register_bonus ? Status::YES : Status::NO;
        $general->save();

        $notify[] = ['success', 'System configuration updated successfully'];

        return back()->withNotify($notify);
    }

    public function logoIcon()
    {
        $pageTitle = 'Logo & Favicon';

        return view('admin.setting.logo_icon', compact('pageTitle'));
    }

    public function logoIconUpdate(Request $request)
    {
        $request->validate([
            'logo' => ['image', new FileTypeValidate(['jpg', 'jpeg', 'png'])],
            'favicon' => ['image', new FileTypeValidate(['png'])],
        ]);
        $path = getFilePath('logoIcon');
        if ($request->hasFile('logo')) {
            try {
                fileUploader($request->logo, $path, filename: 'logo.png');
            } catch (\Exception $exp) {
                $notify[] = ['error', 'Couldn\'t upload the logo'];

                return back()->withNotify($notify);
            }
        }

        if ($request->hasFile('logo_dark')) {
            try {
                fileUploader($request->logo_dark, $path, filename: 'logo_dark.png');
            } catch (\Exception $exp) {
                $notify[] = ['error', 'Couldn\'t upload the logo'];

                return back()->withNotify($notify);
            }
        }

        if ($request->hasFile('favicon')) {
            try {
                fileUploader($request->favicon, $path, filename: 'favicon.png');
            } catch (\Exception $exp) {
                $notify[] = ['error', 'Couldn\'t upload the favicon'];

                return back()->withNotify($notify);
            }
        }
        $notify[] = ['success', 'Logo & favicon updated successfully'];

        return back()->withNotify($notify);
    }

    public function customCss()
    {
        $pageTitle = 'Custom CSS';
        $file = activeTemplate(true).'css/custom.css';
        $fileContent = @file_get_contents($file);

        return view('admin.setting.custom_css', compact('pageTitle', 'fileContent'));
    }
    public function customCssSubmit(Request $request)
    {
        $file = activeTemplate(true).'css/custom.css';
        if (! file_exists($file)) {
            fopen($file, 'w');
        }
        file_put_contents($file, $request->css);
        $notify[] = ['success', 'CSS updated successfully'];

        return back()->withNotify($notify);
    }

    public function customCode()
    {
        $pageTitle = 'Custom Blade File';
        $path = public_path('core/resources/views/components/custom/custom.blade.php');

        // Ensure file exists (optional auto-create)
        if (!file_exists($path)) {
            $dir = dirname($path);
            if (!is_dir($dir)) {
                mkdir($dir, 0755, true); // create directory if missing
            }

            file_put_contents($path, '<!-- Custom Blade Component -->');
        }

        // Read the file content
        $fileContent = file_get_contents($path);

        return view('admin.setting.custom_code', compact('pageTitle', 'fileContent'));
    }
    public function customCodeSubmit(Request $request)
    {
        $path = public_path('core/resources/views/components/custom/custom.blade.php');

        $dir = dirname($path);
        if (!is_dir($dir)) {
            mkdir($dir, 0755, true);
        }

        file_put_contents($path, $request->raw_code);

        $notify[] = ['success', 'Custom code updated successfully'];
        return back()->withNotify($notify);
    }


    public function sitemap()
    {
        $pageTitle = 'Sitemap XML';
        $file = 'sitemap.xml';
        $fileContent = @file_get_contents($file);

        return view('admin.setting.sitemap', compact('pageTitle', 'fileContent'));
    }

    public function sitemapSubmit(Request $request)
    {
        $file = 'sitemap.xml';
        if (! file_exists($file)) {
            fopen($file, 'w');
        }
        file_put_contents($file, $request->sitemap);
        $notify[] = ['success', 'Sitemap updated successfully'];

        return back()->withNotify($notify);
    }

    public function robot()
    {
        $pageTitle = 'Robots TXT';
        $file = 'robots.xml';
        $fileContent = @file_get_contents($file);

        return view('admin.setting.robots', compact('pageTitle', 'fileContent'));
    }

    public function robotSubmit(Request $request)
    {
        $file = 'robots.xml';
        if (! file_exists($file)) {
            fopen($file, 'w');
        }
        file_put_contents($file, $request->robots);
        $notify[] = ['success', 'Robots txt updated successfully'];

        return back()->withNotify($notify);
    }

    

    public function maintenanceMode()
    {
        $pageTitle = 'Maintenance Mode';
        $maintenance = Frontend::where('data_keys', 'maintenance.data')->firstOrFail();

        return view('admin.setting.maintenance', compact('pageTitle', 'maintenance'));
    }

    public function maintenanceModeSubmit(Request $request)
    {
        $request->validate([
            'description' => 'required',
            'image' => ['nullable', new FileTypeValidate(['jpg', 'jpeg', 'png'])],
        ]);
        $general = gs();
        $general->maintenance_mode = $request->status ? Status::ENABLE : Status::DISABLE;
        $general->save();

        $maintenance = Frontend::where('data_keys', 'maintenance.data')->firstOrFail();
        $image = @$maintenance->data_values->image;
        if ($request->hasFile('image')) {
            try {
                $old = $image;
                $image = fileUploader($request->image, getFilePath('maintenance'), getFileSize('maintenance'), $old);
            } catch (\Exception $exp) {
                $notify[] = ['error', 'Couldn\'t upload your image'];

                return back()->withNotify($notify);
            }
        }

        $maintenance->data_values = [
            'description' => $request->description,
            'image' => $image,
        ];
        $maintenance->save();

        $notify[] = ['success', 'Maintenance mode updated successfully'];

        return back()->withNotify($notify);
    }

    public function cookie()
    {
        $pageTitle = 'GDPR Cookie';
        $cookie = Frontend::where('data_keys', 'cookie.data')->firstOrFail();

        return view('admin.setting.cookie', compact('pageTitle', 'cookie'));
    }

    public function cookieSubmit(Request $request)
    {
        $request->validate([
            'short_desc' => 'required|string|max:255',
            'description' => 'required',
        ]);
        $cookie = Frontend::where('data_keys', 'cookie.data')->firstOrFail();
        $cookie->data_values = [
            'short_desc' => $request->short_desc,
            'description' => $request->description,
            'status' => $request->status ? Status::ENABLE : Status::DISABLE,
        ];
        $cookie->save();
        $notify[] = ['success', 'Cookie policy updated successfully'];

        return back()->withNotify($notify);
    }

    public function socialiteCredentials()
    {
        $pageTitle = 'Social Login Credentials';

        return view('admin.setting.social_credential', compact('pageTitle'));
    }

    public function updateSocialiteCredentialStatus($key=null)
    {
        $general = gs();
        $credentials = $general->socialite_credentials;
        try {
            $credentials->$key->status = $credentials->$key->status == Status::ENABLE ? Status::DISABLE : Status::ENABLE;
        } catch (\Throwable $th) {
            abort(404);
        }

        $general->socialite_credentials = $credentials;
        $general->save();

        $notify[] = ['success', 'Status changed successfully'];

        return back()->withNotify($notify);
    }

    public function updateSocialiteCredential(Request $request, $key=null)
    {
        $general = gs();
        $credentials = $general->socialite_credentials;
        try {
            @$credentials->$key->client_id = $request->client_id;
            @$credentials->$key->client_secret = $request->client_secret;
        } catch (\Throwable $th) {
            abort(404);
        }
        $general->socialite_credentials = $credentials;
        $general->save();

        $notify[] = ['success', ucfirst($key).' credential updated successfully'];

        return back()->withNotify($notify);
    }

    public function trustPilot()
    {
        $pageTitle = 'Trustpilot Widget';

        return view('admin.setting.trust_pilot', compact('pageTitle'));
    }

    public function trustPilotInsert(Request $request)
    {
        $request->validate([
            'code' => 'required',
        ]);

        $general = gs();
        $general->trustpilot_widget_code = $request->code;
        $general->save();

        $notify[] = ['success', 'Trustpilot credentials update successfully'];

        return back()->withNotify($notify);
    }
}
