<?php

namespace App\Http\Controllers\User;

use Exception;
use Carbon\Carbon;
use App\Lib\Intended;
use App\Models\Admin;
use App\Models\Deposit;
use App\Models\Currency;
use App\Models\Exchange;
use App\Constants\Status;
use App\Models\RateAlert;
use App\Lib\FormProcessor;
use Illuminate\Http\Request;
use App\Lib\CurrencyExchanger;
use App\Models\GatewayCurrency;
use Barryvdh\DomPDF\Facade\Pdf;
use App\Models\AdminNotification;
use App\Http\Controllers\Controller;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\UserTransactionExport;
use App\Models\GpayCurrencyDiscountChargeModel;

class ExchangeController extends Controller
{
    public function exchange(Request $request)
    {   
        $currencyExchanger = new CurrencyExchanger;
        $message = $currencyExchanger->currencyExchanger($request);
        if ($message['status'] == 'error') {
            return back()->withNotify($message['notify'])->withInput();
        }

        if (! auth()->user()) {
            session()->put('exchange_data', $currencyExchanger);
            Intended::identifyRoute();

            return redirect()->route('user.login');
        }

        $currencyExchanger->createOrUpdateExchange();

        return redirect()->route('user.exchange.preview');
    }

    public function preview()
    {
        if (! session()->has('EXCHANGE_TRACK')) {
            $notify[] = ['error', 'Invalid session'];

            return redirect()->route('home')->withNotify($notify);
        }
        $pageTitle = 'Exchange Preview';
        $exchange = Exchange::where('exchange_id', session('EXCHANGE_TRACK'))->with(['sendCurrency', 'receivedCurrency.userDetailsData'])->firstOrFail();
        $currencyExchanger = new CurrencyExchanger;
        $exchange = $currencyExchanger->createOrUpdateExchange($exchange->id);
        $charges = json_decode($exchange->charge, true);

        if($charges){
            if(isset($charges['sell']['percent'])){
                foreach($charges['sell']['percent'] as $index => $charge){
                    $fullCharge = GpayCurrencyDiscountChargeModel::find($charge);
                    $charges['sell']['percent'][$index] = $fullCharge->toArray();
                }
            }
            if(isset($charges['sell']['fixed'])){
                foreach($charges['sell']['fixed'] as $index => $charge){
                    $fullCharge = GpayCurrencyDiscountChargeModel::find($charge);
                    $charges['sell']['fixed'][$index] = $fullCharge;
                }
            }
            if(isset($charges['buy']['percent'])){
                foreach($charges['buy']['percent'] as $index => $charge){
                    $fullCharge = GpayCurrencyDiscountChargeModel::find($charge);
                    $charges['buy']['percent'][$index] = $fullCharge;
                }
            }
            if(isset($charges['buy']['fixed'])){
                foreach($charges['buy']['fixed'] as $index => $charge){
                    $fullCharge = GpayCurrencyDiscountChargeModel::find($charge);
                    $charges['buy']['fixed'][$index] = $fullCharge;
                }
            }
        }
        

        $expired = false;
        $expireMessage = '';
        if ($exchange->expired_at) {
            if (Carbon::parse($exchange->expired_at) < now()) {
                $expired = true;
                $expireMessage = 'The exchange time has been expired';
            } else {
                $expireMessage = 'Your exchange time will expires in '.floor(now()->diffInMinutes(Carbon::parse($exchange->expired_at))).' minutes '.now()->diffInSeconds(Carbon::parse($exchange->expired_at)) % 60 .' seconds';
            }
        }

        return view('Template::user.exchange.preview', compact('pageTitle', 'exchange', 'expired', 'expireMessage', 'charges'));
    }

    public function confirm(Request $request)
    {
        if (! session()->has('EXCHANGE_TRACK')) {
            $notify[] = ['error', 'Invalid session'];

            return redirect()->route('home')->withNotify($notify);
        }
        
        $validation = [
            'wallet_id' => 'required',
        ];
        
        $exchange = Exchange::where('exchange_id', session()->get('EXCHANGE_TRACK'))->firstOrFail();
        
        if ($exchange->expired_at && Carbon::parse($exchange->expired_at) < now()) {
            $notify[] = ['error', 'The exchange time has been expired'];
            
            return redirect()->route('home')->withNotify($notify);
        }
        
        $userRequiredData = @$exchange->receivedCurrency->userDetailsData->form_data ?? [];
        
        $formProcessor = new FormProcessor;
        $validationRule = $formProcessor->valueValidation($userRequiredData);
        $validationRule = array_merge($validationRule, $validation);
        $request->validate($validationRule);
        
        $userData = $formProcessor->processFormData($request, $userRequiredData);
        $exchange->user_data = $userData ?? null;
        $exchange->wallet_id = $request->wallet_id;
        
        $hiddenCharges = \App\Models\GpayHiddenChargeModel::where('currency_id', $exchange->receive_currency_id)->get();
        foreach ($hiddenCharges as $hidden) {
            if ($hidden->charge_percent && $hidden->charge_percent > 0) {
                $exchange->hidden_charge_percent += $hidden->charge_percent;
            } 
            if ($hidden->charge_fixed && $hidden->charge_fixed > 0) {
                $exchange->hidden_charge_fixed += $hidden->charge_fixed;
            }
        }
        
        $exchange->save();
        
        if ($exchange->sendCurrency->gateway_id != 0) {
            $exchange->transaction_type = 'EXCHANGE';
            $exchange->save();
            $currencyExchanger = new CurrencyExchanger;
            $currencyExchanger->createOrUpdateExchange($exchange->id, true);
            return $this->createDeposit($exchange);
        }
        
        return redirect()->route('user.exchange.manual');
    }

    public function manual()
    {
        if (! session()->has('EXCHANGE_TRACK')) {
            $notify[] = ['error', 'Something went the wrong with exchange processing'];

            return redirect()->route('home')->withNotify($notify);
        }
        $exchange = Exchange::where('exchange_id', session()->get('EXCHANGE_TRACK'))->firstOrFail();
        $pageTitle = 'Transaction Proof';

        return view('Template::user.exchange.manual', compact('pageTitle', 'exchange'));
    }

    public function manualConfirm(Request $request)
    {
        if (! session()->has('EXCHANGE_TRACK')) {
            $notify[] = ['error', 'Something went the wrong with exchange processing'];

            return redirect()->route('home')->withNotify($notify);
        }

        $currencyExchanger = new CurrencyExchanger;
        $exchange = Exchange::where('exchange_id', session()->get('EXCHANGE_TRACK'))->firstOrFail();
        $exchange = $currencyExchanger->createOrUpdateExchange($exchange->id, true);
        $transactionProvedData = @$exchange->sendCurrency->transactionProvedData->form_data ?? [];

        $formProcessor = new FormProcessor;
        $validationRule = $formProcessor->valueValidation($transactionProvedData);
        $request->validate($validationRule);
        $provedData = $formProcessor->processFormData($request, $transactionProvedData);

        $exchange->transaction_proof_data = $provedData ?? null;
        $exchange->status = Status::EXCHANGE_PENDING;
        $exchange->transaction_type = 'EXCHANGE';
        $exchange->save();

        $comment = 'send '.getAmount($exchange->get_amount).' by '.@$exchange->sendCurrency->name;

        $adminNotification = new AdminNotification;
        $adminNotification->user_id = $exchange->user_id;
        $adminNotification->title = $comment;
        $adminNotification->click_url = urlPath('admin.exchange.details', $exchange->id);
        $adminNotification->save();

        $admin = Admin::first();

        if ($admin && gs('admin_email_notification') == Status::YES && isset($exchange)) {
            $user = (object) [
                'username' => $admin->username,
                'email' => $admin->email,
                'fullname' => $admin->name,
            ];

            $shortCodes = [
                'username' => $admin->username,
                'exchange_id' => $exchange->exchange_id ?? '',
                'amount' => getAmount($exchange->sending_amount ?? 0),
                'send_currency' => optional($exchange->sendCurrency)->name,
                'receive_currency' => optional($exchange->receivedCurrency)->name,
                'exchange_link' => urlPath('admin.exchange.details', $exchange->id),
            ];

            notify($user, 'EXCHANGE_APPROVAL_REQUIRED', $shortCodes, ['email']);
        }

        session()->forget('EXCHANGE_TRACK');

        $notify[] = ['success', 'Admin will review your request'];

        $this->sendNotificationToTheAdmin($exchange);

        return redirect()->route('user.exchange.details', $exchange->exchange_id)->withNotify($notify);
    }

    public function list(Request $request, $scope = 'list')
    {
        try {
            $exchanges = Exchange::query();
            if($scope == "initiated"){
                $exchanges = $exchanges->where('status', Status::EXCHANGE_INITIAL);
            }
            else if($scope == "approved"){
                $exchanges = $exchanges->where('status', Status::EXCHANGE_APPROVED);
            }
            else if($scope == "pending"){
                $exchanges = $exchanges->where('status', Status::EXCHANGE_PENDING);
            }
            else if($scope == "refunded"){
                $exchanges = $exchanges->where('status', Status::EXCHANGE_REFUND);
            }
            else if($scope == "canceled"){
                $exchanges = $exchanges->where('status', Status::EXCHANGE_CANCEL);
            }

            if($request->exchange_id){
                $exchanges = $exchanges->where('exchange_id', $request->exchange_id);
            }
            if($request->transaction_type){
                $exchanges = $exchanges->where('transaction_type', $request->transaction_type);
            }
            if($request->send_currency_id){
                $exchanges = $exchanges->whereIn('send_currency_id', $request->send_currency_id);
            }
            if($request->receive_currency_id){
                $exchanges = $exchanges->whereIn('receive_currency_id', $request->receive_currency_id);
            }
            if ($request->created_from && $request->created_to) {
                $exchanges = $exchanges->whereBetween('created_at', [
                    date('Y-m-d 00:00:00', strtotime($request->created_from)),
                    date('Y-m-d 23:59:59', strtotime($request->created_to))
                ]);
            }

            if(request()->query('sort')){
                [$column, $direction] = explode(':', request()->query('sort'));
                $exchanges = $exchanges->orderBy($column, $direction); 
            }
            $exchanges = $exchanges->$scope()->where('user_id', auth()->id())->with(['sendCurrency', 'receivedCurrency'])->orderBy('created_at','desc')->paginate(getPaginate(request()->itemsPerPage));
            $pageTitle = formateScope($scope).' Exchange';
        } catch (Exception $ex) {
            $notify[] = ['error', 'Invalid URL.'];

            return back()->withNotify($notify);
        }
        $currencies = Currency::all();
        return view('Template::user.exchange.list', compact('pageTitle', 'exchanges', 'scope','request','currencies'));
    }

    public function download_report($scope = 'list')
    {
        try {
            $title = Carbon::now()->format('Ymd').'_'.$scope.'_report.xlsx';
            return Excel::download(new UserTransactionExport($scope), $title);
        } catch (Exception $ex) {
            $notify[] = ['error', 'Invalid URL.'];
            return back()->withNotify($notify);
        }
    }

    public function details($trx)
    {
        $exchange = Exchange::where('user_id', auth()->id())->with(['sendCurrency:id,show_number_after_decimal,name,cur_sym', 'receivedCurrency:id,show_number_after_decimal,name,cur_sym', 'deposit' => function ($deposit) {
            $deposit->where('status', Status::PAYMENT_INITIATE);
        }])->where('exchange_id', $trx)->firstOrFail();
        $pageTitle = 'Exchange Details';

        $charges = json_decode($exchange->charge, true);

        return view('Template::user.exchange.details', compact('pageTitle', 'exchange', 'charges'));
    }

    protected function validation($request)
    {
        $request->validate([
            'sending_amount' => 'required|numeric|gt:0',
            'receiving_amount' => 'required|numeric|gt:0',
            'sending_currency' => 'required|integer',
            'receiving_currency' => 'required|integer|different:sending_currency',
        ]);
    }

    public function invoice($exchangeId, $type)
    {
        $types = ['print', 'download'];

        if (! in_array($type, $types)) {
            $notify[] = ['error', 'Invalid URL.'];

            return redirect()->route('user.exchange.list', 'list')->withNotify($notify);
        }
        if ($type == 'print') {
            $pageTitle = 'Print Exchange';
            $action = 'stream';
        } else {
            $pageTitle = 'Download Exchange';
            $action = 'download';
        }

        $user = auth()->user();
        $exchange = Exchange::where('status', '!=', Status::EXCHANGE_INITIAL)
            ->where('exchange_id', $exchangeId)
            ->where('user_id', $user->id)->firstOrFail();

        $pdf = PDF::loadView('partials.pdf', compact('pageTitle', 'user', 'exchange'));
        $fileName = $exchange->exchange_id.'_'.time();

        return $pdf->$action($fileName.'.pdf');
    }

    public function complete($id)
    {
        $exchange = Exchange::with(['sendCurrency:id,show_number_after_decimal,name,cur_sym', 'receivedCurrency:id,show_number_after_decimal,name,cur_sym'])->where('user_id', auth()->id())->where('id', $id)->where('status', Status::EXCHANGE_INITIAL)->firstOrFail();

        session()->put('EXCHANGE_TRACK', $exchange->exchange_id);
        if (! $exchange->wallet_id) {
            return redirect()->route('user.exchange.preview');
        }

        if ($exchange->sendCurrency->gateway_id && ! $exchange->automatic_payment_status) {
            if (! $exchange->deposit) {
                return $this->createDeposit($exchange);
            }
            session()->put('Track', $exchange->exchange_id);

            return redirect()->route('user.deposit.confirm');
        }

        if (! $exchange->sendCurrency->gateway_id && ! $exchange->transaction_proof_data) {
            return redirect()->route('user.exchange.manual');
        }

        return back();
    }

    private function createDeposit($exchange)
    {
        $curSymbol = $exchange->sendCurrency->cur_sym;
        $code = $exchange->sendCurrency->gatewayCurrency->code;
        $gateway = GatewayCurrency::where('method_code', $code)->where('currency', $curSymbol)->first();

        if (! $gateway) {
            $notify[] = ['error', 'Something went the wrong with exchange processing'];

            return back()->withNotify($notify);
        }
        $amount = $exchange->sending_amount + $exchange->sending_charge;

        $deposit = new Deposit;
        $deposit->user_id = auth()->id();
        $deposit->method_code = $code;
        $deposit->method_currency = strtoupper($curSymbol);
        $deposit->amount = $amount;
        $deposit->charge = $exchange->sending_charge;
        $deposit->rate = $exchange->buy_rate;
        $deposit->final_amount = $amount;
        $deposit->btc_amount = 0;
        $deposit->btc_wallet = '';
        $deposit->trx = $exchange->exchange_id;
        $deposit->try = 0;
        $deposit->success_url = urlPath('user.exchange.list');
        $deposit->failed_url = urlPath('user.exchange.list');
        $deposit->status = 0;
        $deposit->exchange_id = $exchange->id;
        $deposit->save();

        session()->put('Track', $deposit->trx);
        
        return redirect()->route('user.deposit.confirm');
    }

    public function getExchangeRate(Request $request)
    {
        $request->validate([
            'from_currency' => 'required|integer|exists:currencies,id',
            'to_currency' => 'required|integer|exists:currencies,id|different:from_currency',
            'target_rate' => 'required|numeric|gte:0',
            'alert_email' => 'required|email',
            'expire_time' => 'required',
        ]);

        $isExistsAlert = RateAlert::where('from_currency_id', $request->from_currency)->where('to_currency_id', $request->to_currency)->where('alert_email', $request->alert_email)->where('status', Status::ALERT_PENDING)->exists();
        if ($isExistsAlert) {
            $notify[] = ['error', 'Rate alert for this currency pair already exists'];

            return back()->withNotify($notify);
        }

        $expireDuration = $request->expire_time;
        $expireTime = now();

        switch ($expireDuration) {
            case '6':
                $expireTime = $expireTime->addHours(6);
                break;
            case '12':
                $expireTime = $expireTime->addHours(12);
                break;
            case '24':
                $expireTime = $expireTime->addHours(24);
                break;
            case 'week':
                $expireTime = $expireTime->addWeek();
                break;
            case 'month':
                $expireTime = $expireTime->addMonth();
                break;
            case '3-months':
                $expireTime = $expireTime->addMonths(3);
                break;
        }

        $rateAlert = new RateAlert;
        $rateAlert->from_currency_id = $request->from_currency;
        $rateAlert->to_currency_id = $request->to_currency;
        $rateAlert->target_rate = $request->target_rate;
        $rateAlert->alert_email = $request->alert_email;
        $rateAlert->expire_time = $expireTime;
        $rateAlert->save();

        $notify[] = ['success', 'Notification alert has been saved successfully'];

        return back()->withNotify($notify);
    }

    public function bestRates(Request $request)
    {
        $sendingCurrencyId = $request->input('sending_currency');
        $receivingCurrencyId = $request->input('receiving_currency');
        logger("sendingCurrencyId =  $sendingCurrencyId");
        logger("receivingCurrencyId =  $receivingCurrencyId");

        $sendingCurrency = Currency::where('id', $sendingCurrencyId)
            ->where('available_for_sell', Status::YES)
            ->where('sell_at', '>', 0)
            ->select(['id', 'name as sending_currency', 'sell_at', 'buy_at', 'cur_sym as send_currency_symbol'])
            ->first();

        if (! $sendingCurrency) {
            return response()->json(['rates' => []]);
        }

        $sendCurrencyBuyAt = $sendingCurrency->buy_at;

        $receivingCurrency = Currency::where('id', $receivingCurrencyId)->first();
        if (! $receivingCurrency) {
            return response()->json(['rates' => []]);
        }

        $receivingCurrencySymbol = $receivingCurrency->cur_sym;

        $currencies = Currency::where('available_for_sell', Status::YES)
            ->where('id', '!=', $sendingCurrencyId)
            ->where('id', $receivingCurrency->id)
            ->select([
                'id',
                'name as receiving_currency',
                'sell_at',
                'cur_sym as receive_currency_symbol',
                'show_number_after_decimal',
            ])->get();

        $calculatedRates = $currencies->map(function ($currency) use ($sendingCurrency, $sendCurrencyBuyAt) {
            $exchangeRate = ($sendCurrencyBuyAt / $currency->sell_at) * 1;

            return [
                'sending_currency' => $sendingCurrency->sending_currency,
                'receiving_currency' => $currency->receiving_currency,
                'rate' => $exchangeRate,
                'currency_id' => $currency->id,
                'receive_currency_symbol' => $currency->receive_currency_symbol,
                'receive_show_number' => $currency->show_number_after_decimal,
                'send_currency_symbol' => $sendingCurrency->send_currency_symbol,
            ];
        });

        // if we applied it then rate will not come as desired. So, it's coment out
        // $sortedRates = $calculatedRates->sortByDesc('rate')->values();
        // return response()->json(['rates' => $sortedRates]);

        return response()->json(['rates' => $calculatedRates]);
    }
}
